<?php
error_log("PDF Worker started for job: " . ($argv[1] ?? 'unknown'));

require('../tfpdf/tfpdf.php');
require_once 'category_wise_fields.php';

if (empty($argv[1])) {
    error_log("PDF Worker: No job ID provided");
    exit(1);
}

$jobId = $argv[1];
$jobFile = "jobs/{$jobId}.json";

if (!file_exists($jobFile)) {
    error_log("PDF Worker: Job file not found: {$jobFile}");
    exit(1);
}

$jobData = json_decode(file_get_contents($jobFile), true);
$jobData['status'] = 'processing';
$jobData['started_at'] = time();
file_put_contents($jobFile, json_encode($jobData));

$authKey = 'pat-na1-62de1cbe-0ad2-46e2-b982-10c693e2eaf2';
$tableId = $jobData['tableId'];
$hsIds = $jobData['hsIds'];
$parentFolderId = '113972357259';
$tempFolder = 'submissions_folder';
$zipFilename = "submissions_list.zip";
$csvFilename = "uploaded_files.csv";

// Function to clean and encode text for PDF
function cleanTextForPdf($text) {
    if (empty($text)) return $text;
    
    // Decode any HTML entities first
    $text = html_entity_decode($text, ENT_QUOTES, 'UTF-8');
    
    // Replace tab characters with spaces
    $text = str_replace("\t", "    ", $text); // Replace tabs with 4 spaces
    
    // Ensure proper UTF-8 encoding
    if (!mb_check_encoding($text, 'UTF-8')) {
        $text = mb_convert_encoding($text, 'UTF-8', 'auto');
    }
    
    return $text;
}


try {
    error_log("PDF Worker: Starting processing for {$tableId} with " . count($hsIds) . " IDs");

    // Fetch HubDB rows
    $url = "https://api.hubapi.com/cms/v3/hubdb/tables/{$tableId}/rows?limit=10000";
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 60);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        "Authorization: Bearer $authKey",
        "Content-Type: application/json"
    ]);
    $response = curl_exec($ch);
    if (curl_error($ch)) throw new Exception('HubDB API error: ' . curl_error($ch));
    curl_close($ch);

    $data = json_decode($response, true);
    $matchedRows = [];
    foreach ($data['results'] ?? [] as $row) {
        if (in_array($row['id'], $hsIds)) $matchedRows[] = $row;
    }

    if (empty($matchedRows)) throw new Exception('No matching HubDB rows found for provided hsIds');

    error_log("PDF Worker: Found " . count($matchedRows) . " matching rows");

    if (!is_dir($tempFolder)) mkdir($tempFolder, 0777, true);

    $csvHandle = fopen($csvFilename, 'w');
    fputcsv($csvHandle, ['Row ID', 'Submission Name', 'PDF Filename', 'HubSpot URL']);

    $uploadedFiles = [];
    $processedCount = 0;

    foreach ($matchedRows as $row) {
        $processedCount++;
        error_log("PDF Worker: Processing row {$processedCount}/" . count($matchedRows) . " - ID: " . $row['id']);
        
        $values = $row['values'];
        $category = $values['category'] ?? '';
        $raw = !empty($values['raw']) ? json_decode($values['raw'], true) : [];
        $question_list = getQuestionList($category);

        if (!empty(trim($values['devices_software_app_name']))) {
            $pdf_file_name = trim($values['devices_software_app_name']);
        } else if (!empty(trim($raw['company_name']))) {
            $pdf_file_name = trim($raw['company_name']);
        } else if (!empty(trim($raw['institution_name']))) {
            $pdf_file_name = trim($raw['institution_name']);
        } else if (!empty(trim($category))) {
            $pdf_file_name = trim($category);
        } else {
            $pdf_file_name = 'Submission';
        }

        $product_name = preg_replace('/[^A-Za-z0-9_\-]/', '_', $pdf_file_name);
        
        

        // ---------------------------
        // Create PDF using tFPDF
        // ---------------------------
        $pdf = new tFPDF();
        $pdf->AddPage();

        // Add DejaVu fonts
        $pdf->AddFont('DejaVu','','DejaVuSans.ttf', true);
        $pdf->AddFont('DejaVu','B','DejaVuSans-Bold.ttf', true);
        $pdf->AddFont('DejaVu','I','DejaVuSans-Oblique.ttf', true);
        $pdf->AddFont('DejaVu','BI','DejaVuSans-BoldOblique.ttf', true);

        // Clean the main title
        $clean_pdf_file_name = cleanTextForPdf($pdf_file_name);
        
        $pdf->SetFont('DejaVu','B',14);
        $pdf->Cell(0, 10, $clean_pdf_file_name, 0, 1, 'C');
        $pdf->Ln(5);
        $pdf->SetFont('DejaVu','',12);

        foreach ($question_list as $q) {
            $label = cleanTextForPdf($q['label'] ?? '');
            $sub_label = cleanTextForPdf($q['sub-label'] ?? '');
            $name = $q['name'] ?? '';
            $type = $q['type'] ?? '';
            $value = $raw[$name] ?? '';

            $pdf->Ln(5);
            $pdf->SetFont('DejaVu','B',14);
            $pdf->MultiCell(0, 7, $label, 0, 1);
            $pdf->Ln(3);

            if (!empty($sub_label)) {
                $pdf->SetFont('DejaVu','BI',14);
                $pdf->MultiCell(0, 7, $sub_label, 0, 1);
                $pdf->Ln(3);
                $pdf->SetFont('DejaVu','B',12);
            }

            if ($type === 'image' && $value !== '' && $value !== '|') {
                $files = explode("|", $value);
                foreach ($files as $file) {
                    $file = trim($file);
                    if (!empty($file)) {
                        $url = "https://5486117.fs1.hubspotusercontent-na1.net/hubfs/5486117/Submission%20Files/{$row['id']}/{$file}";
                        $pdf->SetTextColor(0,0,255);
                        $pdf->Write(6, $file, $url);
                        $pdf->Ln(6);
                        $pdf->SetTextColor(0,0,0);
                    }
                }
            } else if ($type === 'array' && $value !== '' && is_array($value)) {
                foreach ($value as $val) {
                    $pdf->SetFont('DejaVu','',12);
                    $cleanVal = cleanTextForPdf($val);
                    $pdf->MultiCell(0, 6, "• " . $cleanVal, 0, 1); // bullet point
                    $pdf->Ln(2);
                }
            } else if ($value) {
                $pdf->SetFont('DejaVu','',12);
                $cleanValue = cleanTextForPdf($value);
                
                // Handle multiline text that might contain tabs
                $lines = explode("\n", $cleanValue);
                foreach ($lines as $line) {
                    $pdf->MultiCell(0, 6, $line, 0, 1);
                }
                $pdf->Ln(2);
            } else {
                $pdf->SetFont('DejaVu','',12);
                $pdf->MultiCell(0, 6, 'N/A', 0, 1);
                $pdf->Ln(2);
            }
        }

        // ---------------------------
        // Create category folder
        // ---------------------------
        $categoryFolderName = preg_replace('/[^A-Za-z0-9_\-]/', '_', $category ?: 'Uncategorized');
        $categoryFolderPath = $tempFolder . '/' . $categoryFolderName;
        if (!is_dir($categoryFolderPath)) mkdir($categoryFolderPath, 0777, true);

        // Create product folder inside category
        $productFolderName = preg_replace('/[^A-Za-z0-9_\-]/', '_', $pdf_file_name);
        $productFolderPath = $categoryFolderPath . '/' . $productFolderName;
        if (!is_dir($productFolderPath)) mkdir($productFolderPath, 0777, true);

        $filename = $productFolderPath . '/submission_' . $product_name . '.pdf';
        $pdf->Output('F', $filename);
        
        
        
        
        
        
if(false)
{
    // ---------------------------
    // Create DOCX (No Composer)
    // ---------------------------
    $docXml = '<?xml version="1.0" encoding="UTF-8" standalone="yes"?>
    <w:document xmlns:w="http://schemas.openxmlformats.org/wordprocessingml/2006/main">
      <w:body>';
    
    // Title
    //$docXml .= '<w:p><w:r><w:t>' . htmlspecialchars(cleanTextForPdf($pdf_file_name)) . '</w:t></w:r></w:p>';
    //$docXml .= '<w:p><w:r><w:br/></w:r></w:p>';
    
    foreach ($question_list as $q) {
        $label = cleanTextForPdf($q['label'] ?? '');
        $sub_label = cleanTextForPdf($q['sub-label'] ?? '');
        $name = $q['name'] ?? '';
        $type = $q['type'] ?? '';
        $value = $raw[$name] ?? '';
    
        // Label
        $docXml .= '<w:p><w:r><w:rPr><w:b/><w:sz w:val="28"/><w:szCs w:val="28"/></w:rPr><w:t xml:space="preserve">'
             . htmlspecialchars($label) .
             '</w:t></w:r></w:p>';
    
    
        // Sub-label
        if (!empty($sub_label)) {
           $docXml .= '<w:p><w:r><w:rPr><w:i/><w:sz w:val="24"/><w:szCs w:val="24"/></w:rPr><w:t xml:space="preserve">'
             . htmlspecialchars($sub_label) .
             '</w:t></w:r></w:p>';
        }
    
        // Value
        if ($type === 'image' && $value !== '' && $value !== '|') {
            $files = explode("|", $value);
            foreach ($files as $file) {
                $file = trim($file);
                if (!empty($file)) {
                    $url = "https://5486117.fs1.hubspotusercontent-na1.net/hubfs/5486117/Submission%20Files/{$row['id']}/{$file}";
                    $docXml .= '<w:p><w:r><w:t xml:space="preserve">' . htmlspecialchars($file . " - " . $url) . '</w:t></w:r></w:p>';
                }
            }
        } elseif (is_array($value)) {
            foreach ($value as $val) {
                $docXml .= '<w:p><w:r><w:t xml:space="preserve">• ' . htmlspecialchars(cleanTextForPdf($val)) . '</w:t></w:r></w:p>';
            }
        } elseif (!empty($value)) {
            $lines = explode("\n", cleanTextForPdf($value));
            foreach ($lines as $line) {
                $docXml .= '<w:p><w:r><w:t xml:space="preserve">' . htmlspecialchars($line) . '</w:t></w:r></w:p>';
            }
        } else {
            $docXml .= '<w:p><w:r><w:t>N/A</w:t></w:r></w:p>';
        }
    
        $docXml .= '<w:p><w:r><w:br/></w:r></w:p>';
    }
    
    $docXml .= '</w:body></w:document>';
    
    // ---------------------------
    // Create .docx structure
    // ---------------------------
    $filename = $productFolderPath . '/submission_' . $product_name . '.docx';
    $tempDocxDir = sys_get_temp_dir() . '/docx_' . uniqid();
    mkdir($tempDocxDir . '/word/_rels', 0777, true);
    mkdir($tempDocxDir . '/_rels', 0777, true);
    
    // document.xml
    file_put_contents($tempDocxDir . '/word/document.xml', $docXml);
    
    // styles.xml (minimal)
    $styles = '<?xml version="1.0" encoding="UTF-8" standalone="yes"?>
    <w:styles xmlns:w="http://schemas.openxmlformats.org/wordprocessingml/2006/main">
    <w:style w:type="paragraph" w:default="1" w:styleId="Normal">
    <w:name w:val="Normal"/>
    </w:style>
    </w:styles>';
    file_put_contents($tempDocxDir . '/word/styles.xml', $styles);
    
    // [Content_Types].xml
    $contentTypes = '<?xml version="1.0" encoding="UTF-8"?>
    <Types xmlns="http://schemas.openxmlformats.org/package/2006/content-types">
    <Default Extension="rels" ContentType="application/vnd.openxmlformats-package.relationships+xml"/>
    <Default Extension="xml" ContentType="application/xml"/>
    <Override PartName="/word/document.xml" ContentType="application/vnd.openxmlformats-officedocument.wordprocessingml.document.main+xml"/>
    <Override PartName="/word/styles.xml" ContentType="application/vnd.openxmlformats-officedocument.wordprocessingml.styles+xml"/>
    </Types>';
    file_put_contents($tempDocxDir . '/[Content_Types].xml', $contentTypes);
    
    // Relationships
    $rels = '<?xml version="1.0" encoding="UTF-8"?>
    <Relationships xmlns="http://schemas.openxmlformats.org/package/2006/relationships">
    <Relationship Id="rId1" Type="http://schemas.openxmlformats.org/officeDocument/2006/relationships/officeDocument" Target="word/document.xml"/>
    </Relationships>';
    file_put_contents($tempDocxDir . '/_rels/.rels', $rels);
    
    // word/_rels/document.xml.rels
    $wordRels = '<?xml version="1.0" encoding="UTF-8"?>
    <Relationships xmlns="http://schemas.openxmlformats.org/package/2006/relationships" />';
    file_put_contents($tempDocxDir . '/word/_rels/document.xml.rels', $wordRels);
    
    // ---------------------------
    // Zip into DOCX
    // ---------------------------
    $zip = new ZipArchive();
    $zip->open($filename, ZipArchive::CREATE | ZipArchive::OVERWRITE);
    $rootPath = realpath($tempDocxDir);
    $files = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($rootPath, RecursiveDirectoryIterator::SKIP_DOTS),
        RecursiveIteratorIterator::LEAVES_ONLY
    );
    foreach ($files as $file) {
        if (!$file->isDir()) {
            $filePath = $file->getRealPath();
            $relativePath = substr($filePath, strlen($rootPath) + 1);
            $zip->addFile($filePath, $relativePath);
        }
    }
    $zip->close();
    
    // Cleanup temp XML folder
    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($tempDocxDir, RecursiveDirectoryIterator::SKIP_DOTS),
        RecursiveIteratorIterator::CHILD_FIRST
    );
    foreach ($iterator as $file) {
        $file->isDir() ? rmdir($file->getRealPath()) : unlink($file->getRealPath());
    }
    rmdir($tempDocxDir);
    //end doc
}        
        
        
        
        
        
        
        
        

        // ---------------------------
        // Upload to HubSpot
        // ---------------------------
        $subfolderId = createOrGetHubspotFolder($authKey, '' . $row['id'], $parentFolderId);
        $pdfData = base64_encode(file_get_contents($filename));
        $url = uploadPdfToHubspot($authKey, basename($filename), $pdfData, $subfolderId);

        fputcsv($csvHandle, [$row['id'], $product_name, basename($filename), $url ?? '']);
        $uploadedFiles[] = [
            'rowId' => $row['id'],
            'submissionName' => $product_name,
            'pdf' => basename($filename),
            'hubspotUrl' => $url ?? null
        ];

        error_log("PDF Worker: Completed row {$row['id']} - PDF: " . basename($filename));
    }

    fclose($csvHandle);

    // ---------------------------
    // Create ZIP
    // ---------------------------
    $zip = new ZipArchive();
    if ($zip->open($zipFilename, ZipArchive::CREATE | ZipArchive::OVERWRITE) === TRUE) {
        $rootPath = realpath($tempFolder);

        $files = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($rootPath, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::LEAVES_ONLY
        );

        foreach ($files as $file) {
            if (!$file->isDir()) {
                $filePath = $file->getRealPath();
                $relativePath = substr($filePath, strlen($rootPath) + 1);
                $zip->addFile($filePath, $relativePath);
            }
        }

        $zip->close();
        error_log("PDF Worker: ZIP file created: {$zipFilename}");
    }

    // ---------------------------
    // Remove temp folder
    // ---------------------------
    function rrmdir($dir) {
        if (!is_dir($dir)) return;
        $files = array_diff(scandir($dir), ['.', '..']);
        foreach ($files as $file) {
            $path = "$dir/$file";
            is_dir($path) ? rrmdir($path) : unlink($path);
        }
        rmdir($dir);
    }
    rrmdir($tempFolder);

    // ---------------------------
    // Finish job
    // ---------------------------
    $jobData['status'] = 'completed';
    $jobData['completed_at'] = time();
    $jobData['result'] = [
        'uploadedFiles' => $uploadedFiles,
        'zipFile' => $zipFilename,
        'csvFile' => $csvFilename,
        'processedRows' => count($uploadedFiles)
    ];
    file_put_contents($jobFile, json_encode($jobData));
    error_log("PDF Worker: Job {$jobId} completed successfully");

} catch (Exception $e) {
    $jobData['status'] = 'error';
    $jobData['error'] = $e->getMessage();
    $jobData['completed_at'] = time();
    file_put_contents($jobFile, json_encode($jobData));
    error_log("PDF Worker: Job {$jobId} failed - " . $e->getMessage());
}

// ---------------------------
// HubSpot functions (unchanged)
// ---------------------------
function createOrGetHubspotFolder($authKey, $folderName, $parentId) {
    $folderId = getHubspotFolderIdByName($authKey, $folderName, $parentId);
    if ($folderId) return $folderId;

    $res = hubspotRequest('POST', 'https://api.hubapi.com/files/v3/folders', $authKey, [
        'name' => $folderName,
        'parentFolderId' => (string)$parentId,
        'options' => ['access' => 'PUBLIC_NOT_INDEXABLE']
    ]);
    return $res['id'] ?? null;
}

function getHubspotFolderIdByName($authKey, $folderName, $parentId) {
    $res = hubspotRequest('GET', "https://api.hubapi.com/filemanager/api/v3/folders?parentFolderId={$parentId}", $authKey);
    foreach ($res['results'] ?? [] as $f) {
        if ($f['name'] === $folderName) return $f['id'];
    }
    return null;
}

function uploadPdfToHubspot($authKey, $filename, $pdfData, $folderId) {
    if (!$folderId) { error_log("HubSpot upload failed: folder ID is null"); return false; }

    $tempFile = tempnam(sys_get_temp_dir(), 'pdf_');
    file_put_contents($tempFile, base64_decode($pdfData));

    $boundary = uniqid();
    $delimiter = '-------------' . $boundary;

    $postData = buildMultipartData([
        'file' => ['path' => $tempFile, 'filename' => $filename],
        'folderId' => $folderId,
        'options' => json_encode([
            'access' => 'PUBLIC_NOT_INDEXABLE',
            'overwrite' => true,
            'duplicateValidationStrategy' => 'NONE',
            'duplicateValidationScope' => 'EXACT_FOLDER'
        ])
    ], $delimiter);

    $ch = curl_init('https://api.hubapi.com/files/v3/files');
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => $postData,
        CURLOPT_HTTPHEADER => [
            "Authorization: Bearer $authKey",
            "Content-Type: multipart/form-data; boundary=$delimiter"
        ]
    ]);
    $res = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    unlink($tempFile);

    if ($httpCode >= 400) {
        error_log("HubSpot upload failed ($httpCode): $res");
        return false;
    }

    return json_decode($res, true)['url'] ?? null;
}

function hubspotRequest($method, $url, $authKey, $data = null) {
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        "Authorization: Bearer $authKey",
        "Content-Type: application/json"
    ]);

    if ($method === 'POST' && $data) {
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    }

    $res = curl_exec($ch);
    if (curl_errno($ch)) throw new Exception("CURL error: " . curl_error($ch));

    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($httpCode >= 400) throw new Exception("HubSpot API error ($httpCode): $res");
    return json_decode($res, true);
}

function buildMultipartData($fields, $boundary) {
    $data = '';
    foreach ($fields as $name => $content) {
        $data .= "--$boundary\r\n";
        if (isset($content['path'])) {
            $fileData = file_get_contents($content['path']);
            $filename = $content['filename'];
            $data .= "Content-Disposition: form-data; name=\"$name\"; filename=\"$filename\"\r\n";
            $data .= "Content-Type: application/pdf\r\n\r\n";
            $data .= $fileData . "\r\n";
        } else {
            $data .= "Content-Disposition: form-data; name=\"$name\"\r\n\r\n";
            $data .= $content . "\r\n";
        }
    }
    $data .= "--$boundary--";
    return $data;
}
?>